<?php if ( ! class_exists( 'GhostPool_Importer' ) ) {
	class GhostPool_Importer {

		private static $instance;
		private static $demo_data = array();
		public $error = '';
		public $messages = array();
		public $data_imported = false;
		public $posts_imported = array();
		public $processes = 0;
		public $done_processes = 0;
		public $progress_pid = null;

		function __construct() {

			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
			add_action( 'wp_ajax_ghostpool_single_import', array( $this, 'do_ajax' ) );
			add_action( 'wp_ajax_ghostpool_set_as_home', array( $this, 'set_as_homepage' ) );
			add_action( 'admin_init', array( $this, 'do_import' ), 12 );
			
			if ( isset( $_GET['ghostpool_single_import'] ) && $_GET['ghostpool_single_import'] ) {
				if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
					add_filter( 'wp_redirect', function () {
						return false;
					});
				}
			}

			$this->add_initial_demo_sets();
					
		}

		/**
		* Enqueue style and scripts
		*
		*/
		public function enqueue_scripts() {
			
			if ( isset( $_GET['page'] ) && 'aardvark-theme-setup' === $_GET['page'] ) {
				
				wp_enqueue_script( 'jquery-ui-tooltip' );
				
				wp_enqueue_style( 'ghostpool-importer-css', get_template_directory_uri() . '/lib/framework/theme-setup/importer/assets/importer.css', array(), AARDVARK_THEME_VERSION );
				
				wp_enqueue_script( 'ghostpool-importer-js', get_template_directory_uri() . '/lib/framework/theme-setup/importer/assets/importer.js', array( 'jquery', 'jquery-ui-tooltip' ), AARDVARK_THEME_VERSION, true );
				
			}
			
		}
		
		/**
		* Load demo data
		*
		*/
		public function add_initial_demo_sets() {
			include_once( get_theme_file_path( '/lib/framework/theme-setup/importer/import-data.php' ) );
			$demo_data = ghostpool_importer_data();																
			self::add_demo_sets( $demo_data );
		}

		/**
		* Add multiple demo sets
		*
		*/
		static function add_demo_sets( $data ) {
			self::$demo_data = self::$demo_data + $data;
		}
	
		public static function getInstance() {
			if ( is_null( self::$instance ) ) {
				self::$instance = new self();
			}
			return self::$instance;
		}

		/**
		* Add a demo set
		*	
		*/
		static function add_demo_set( $slug, $data ) {
			self::$demo_data[ $slug ] = $data;
		}

		/**
		* Set as homepage
		*
		*/	
		public function set_as_homepage() {
		
			if ( session_id() ) {
				session_write_close();
			}
		
			check_ajax_referer( 'import_nonce', 'security' );

			if ( ! current_user_can( 'manage_options' ) ) {
				wp_send_json_error( array(
					'message' => esc_html__( 'We&apos;re sorry, something went wrong.', 'aardvark' ),
				) );
				exit;
			}

			if ( isset( $_POST['pid'] ) ) {
				$post_id = $_POST['pid'];
				if ( get_post_status( $post_id ) == 'publish' ) {
					if ( 'page' == get_post_type( $post_id ) ) {
						
						// Set page as homepage
						update_option( 'page_on_front', $post_id );
						update_option( 'show_on_front', 'page' );
				
						wp_send_json_success( array(
							'message' => esc_html__( 'Successfully set as homepage.', 'aardvark' ),
						) );
						exit;
					}
				}
			}
			wp_send_json_success( array(
				'message' => esc_html__( 'An error occurred setting this page as your homepage.', 'aardvark' ),
			) );
			exit;

		}

		/**
		* Ajax
		*
		*/
		public function do_ajax() {
			if ( session_id() ) {
				session_write_close();
			}

			check_ajax_referer( 'import_nonce', 'security' );

			if ( ! current_user_can( 'manage_options' ) ) {
				wp_send_json_error( array(
					'message' => $this->set_error_message( esc_html__( 'We&apos;re sorry, the demo failed to import.', 'aardvark' ) ),
				) );
				exit;
			}

			if ( ! isset( $_POST['options'] ) ) {
				wp_send_json_error( array(
					'message' => $this->set_error_message( esc_html__( 'Something went wrong. Please try again.', 'aardvark' ) ),
				) );
				exit;
			}

			$data = array();

			parse_str( $_POST['options'], $data );

			if ( ! isset( $data['import_demo'] ) ) {
				wp_send_json_error( array(
					'message' => $this->set_error_message( esc_html__( 'Something went wrong with the data sent. Please try again.', 'aardvark' ) ),
				) );
				exit;
			}

			$demos   = self::get_demo_sets();
			$current_set = $data['import_demo'];

			if ( ! array_key_exists( $current_set, $demos ) ) {
				wp_send_json_error( array(
					'message' => $this->set_error_message( esc_html__( 'Something went wrong with the data sent. Please try again.', 'aardvark' ) ),
				) );
				exit;
			}

			$set_data = $demos[ $current_set ];
			$progress_pid = intval( $_POST['pid'] );

			/* If we are checking progress */
			if ( isset( $_POST['check_progress'] ) ) {
				$progress = $this->get_progress( $progress_pid );
				if ( $progress ) {
					wp_send_json_success( array(
						'message' => $progress['text'],
						'progress' => $progress['progress'],
					) );
				}
				exit;
			}

			$response = $this->process_import( array(
				'set_data' => $set_data,
				'pid' => $progress_pid,
				'data' => $data,
			) );

			if ( is_wp_error( $response ) ) {
				wp_send_json_error( array(
					'message' => $this->set_error_message(
						'<p>' . esc_html__( 'There was an error in the import process. Try importing again.', 'aardvark' ) .
						'</p>' . $response->get_error_message()
					),
					'debug'   => implode( ',', $this->messages ),
				) );
				exit;
			}

			$response['debug'] = implode( ',', $this->messages );
			$response['message'] = $this->set_success_message( $response['message'] );

			wp_send_json_success( $response );

		}

		private function set_error_message( $msg ) {
			return $msg;
		}

		/**
		* Retrieve the demo sets
		*	
		*/
		static function get_demo_sets() {
			return self::$demo_data;
		}

		public function get_progress( $post_id ) {
			$data = get_transient( 'ghostpool_import_' . floatval( $post_id ) );
			return $data;
		}
	
		/**
		* Process all the import steps
		*
		* @param array $options
		*
		* @return array|WP_Error
		*/
		public function process_import( $options ) {

			$imported = false;
			$content_imported = false;

			$set_data = $options['set_data'];
			$progress_pid = $options['pid'];
			$this->progress_pid = $progress_pid;
			$data = $options['data'];

			// Importer classes
			if ( ! defined( 'WP_LOAD_IMPORTERS' ) ) {
				define( 'WP_LOAD_IMPORTERS', true );
			}

			if ( ! class_exists( 'WP_Importer' ) ) {
				require_once ABSPATH . 'wp-admin/includes/class-wp-importer.php';
			}

			if ( ! class_exists( 'WP_Import' ) ) {
				require_once get_theme_file_path( '/lib/framework/theme-setup/importer/wordpress-importer.php' );
			}

			if ( ! class_exists( 'WP_Importer' ) || ! class_exists( 'WP_Import' ) ) {
				return new WP_Error( '__k__',  esc_html__( 'Something went wrong. Please try again.', 'aardvark' ) );
			}

			$this->processes = count( $data ) + 1;
			$this->done_processes = 0;

			// Activate required plugins
			if ( isset( $data['activate_plugins'] ) ) {
		
				$this->processes += count( $set_data['plugins'] ) - 1;

				$this->set_progress( $progress_pid, array(
					'text'     => esc_html__( 'Activating any required plugins...', 'aardvark' ),
					'plugins' =>  isset( $set_data['plugins'] ) && ! empty( $set_data['plugins'] ) ? $set_data['plugins'] : array(),
				) );

				$this->activate_plugins( $set_data );
				$this->done_processes ++;
			}

			// Import pages xml
			if ( isset( $data['import_page'] ) && isset( $set_data['page'] ) ) {

				$this->set_progress( $progress_pid, array(
					'text' => esc_html__( 'Importing Pages...', 'aardvark' ),
				) );

				$imported = true;
				$content_imported = true;

				if ( is_array( $data['import_page'] ) ) {
					$the_page = $data['import_page'][0];
				} else {
					$the_page = $data['import_page'];
				}
				$the_page = ucwords( str_replace( array( '-', '_' ), ' ', $the_page ) );

				$file_path = $set_data['page'] . '.xml';
				$this->import_content( $file_path, true );
			
				$this->get_demo_homepage();
			
				$this->messages[] = sprintf( esc_html__( 'Installed page: %s', 'aardvark' ), $the_page );
				$this->done_processes ++;
			}

			// Import widgets
			if ( isset( $data['import_widgets'] ) ) {

				$imported = true;

				// Import sidebars		
				if ( isset( $set_data['sidebars'] ) ) {
					$this->set_progress( $progress_pid, array(
						'text' => esc_html__( 'Importing Sidebars...', 'aardvark' ),
					) );					
					$this->import_sidebars( $set_data['sidebars'] );	
					$this->messages[] = esc_html__( 'Imported Sidebars!', 'aardvark' );	
				}

				// Import widgets
				if ( isset( $set_data['widgets'] ) ) {
					$this->set_progress( $progress_pid, array(
						'text' => esc_html__( 'Importing Widgets...', 'aardvark' ),
					) );
					$widget_path = $set_data['widgets'];
					$file_path = get_theme_file_path( '/lib/framework/theme-setup/importer/demo-files/' . $widget_path . '.txt' );
					if ( file_exists( $file_path ) ) {
						$file_data = ghostpool_fs_get_contents( $file_path );
						if ( $file_data ) {
							$this->import_widget_data( $file_data );
							$this->messages[] = esc_html__( 'Imported Widgets!', 'aardvark' );
						}
					}	
				}
								
				$this->done_processes ++;
				
			}

			// Check options
			if ( isset( $data['import_options'] ) && isset( $set_data['options'] ) ) {

				$imported = true;

				$this->set_progress( $progress_pid, array(
					'text' => esc_html__( 'Importing Theme Options...', 'aardvark' ),
				) );

				$this->import_options( $set_data['options'] );

				$this->messages[] = esc_html__( 'Imported Theme Options!', 'aardvark' );
				$this->done_processes ++;
			}

			// Check Ultimate Reviewer settings
			if ( isset( $data['import_gpur_settings'] ) && isset( $set_data['gpur_settings'] ) ) {

				$imported = true;

				$this->set_progress( $progress_pid, array(
					'text' => esc_html__( 'Importing Ultimate Reviewer Settings...', 'aardvark' ),
				) );

				$this->import_gpur_settings( $set_data['gpur_settings'] );

				$this->messages[] = esc_html__( 'Imported Ultimate Reviewer Settings!', 'aardvark' );
				$this->done_processes ++;
			}
			
			// Check LayerSlider
			if ( isset( $data['import_layerslider'] ) && isset( $set_data['layerslider'] ) ) {
			
				if ( ! class_exists( 'ZipArchive' ) ) {
					return;
				}
				
				$imported = true;

				$this->set_progress( $progress_pid, array(
					'text' => esc_html__( 'Importing LayerSlider...', 'aardvark' ),
				) );
			
				$sliders = (array) $set_data['layerslider'];
				if ( ! empty( $sliders ) ) {
					foreach( $sliders as $slider ) {					
						$file = get_theme_file_path( '/lib/framework/theme-setup/importer/demo-files/' . $slider . '.zip' );
						$this->import_layerslider( $file );
						$this->messages[] = esc_html__( 'Imported LayerSliders!', 'aardvark' );
					}
				}
				$this->done_processes ++;
			}

			// Check BuddyPress data
			if ( isset( $data['import_buddypress'] ) && isset( $set_data['buddypress'] ) ) {

				if ( ! function_exists( 'bp_is_active' ) ) {
					return;
				}
		
				$imported = true;

				$this->set_progress( $progress_pid, array(
					'text' => esc_html__( 'Importing BuddyPress Data...', 'aardvark' ),
				) );
			
				$this->import_buddypress_data();
			
				$this->messages[] = esc_html__( 'Imported BuddyPress Data!', 'aardvark' );
				$this->done_processes ++;
			}

			// Check BuddyPress profile fields
			if ( isset( $data['import_bp_fields'] ) && isset( $set_data['bp_fields'] ) ) {

				if ( ! function_exists( 'bp_is_active' ) || ! bp_is_active( 'xprofile' ) ) {
					return;
				}
			
				$imported = true;

				$this->set_progress( $progress_pid, array(
					'text' => esc_html__( 'Importing BuddyPress Profile Fields...', 'aardvark' ),
				) );

				$this->import_bp_fields( $set_data['bp_fields'] );
				$this->messages[] = esc_html__( 'Imported BuddyPress Profile Fields!', 'aardvark' );
				$this->done_processes ++;
			}

			// Check membership levels
			if ( isset( $data['import_levels'] ) && isset( $set_data['levels'] ) ) {

				if ( ! defined( 'PMPRO_VERSION' ) ) {
					return;
				}
		
				$imported = true;

				$this->set_progress( $progress_pid, array(
					'text' => esc_html__( 'Importing Membership Levels...', 'aardvark' ),
				) );
			
				$this->import_pmpro_levels();
			
				$this->messages[] = esc_html__( 'Imported Membership Levels!', 'aardvark' );
				$this->done_processes ++;
				
			}
			
			// Import Youzer options
			if ( isset( $set_data['youzer'] ) ) {

				if ( ! function_exists( 'youzer' ) ) {
					return;
				}
		
				$imported = true;

				$this->set_progress( $progress_pid, array(
					'text' => esc_html__( 'Importing Youzer Options...', 'aardvark' ),
				) );
			
				$this->import_youzer_options();
			
				$this->messages[] = esc_html__( 'Imported Youzer Options!', 'aardvark' );
				$this->done_processes ++;
				
			}
										
			// Check extras
			if ( isset( $set_data['extra'] ) && is_array( $set_data['extra'] ) ) {

				foreach( $set_data['extra'] as $extra ) {
				
					if ( ! isset( $data[ 'import_' . $extra['id'] ] ) || ! isset( $extra['id'] ) ) {
						continue;
					}
				
					$imported = true;
				
					if ( 'menus' != $extra['id'] ) {
						$content_imported = true;
					}
				
					$this->set_progress( $progress_pid, array(
						'text' => esc_html__( 'Importing', 'aardvark' ) . ' ' . ucfirst( $extra['id'] ) . '...',
					) );

					$ok_to_import = true;

					if ( $ok_to_import ) {
						$file_path = $extra['data'] . '.xml';
						$this->import_content( $file_path, true );
						$this->messages[] = sprintf( esc_html__( '%s complete', 'aardvark' ), $extra['name'] );
					}
				
					if ( 'menus' == $extra['id'] ) {
						$this->import_menu_location( $extra['locations'] );
					}
								
					$this->done_processes ++;
				}
			}

			// Import content
			if ( $content_imported ) {
				$this->processes ++;
				$this->post_process_posts( $set_data );
				$this->done_processes ++;
			}

			// Import messages
			$success_message = '<h4 class="gp-import-success">' . esc_html__( 'Your import was successful!', 'aardvark' ) . '</h4>';

			if ( ! empty( $this->posts_imported ) ) {
								
				$this->posts_imported = array_reverse( $this->posts_imported, true );
				foreach( $this->posts_imported as $post_id => $item ) {
					$success_message .= '<div class="gp-import-summary">' .
						'<p>' . get_the_title( $post_id ) . '</p>' .
						'<a href="#" class="gp-set-as-home button button-primary" title="' . esc_html__( 'Set as Homepage', 'aardvark' ) . '" data-pid="' . $post_id . '">' . esc_html__( 'Set as Homepage', 'aardvark' ) . '</a>' .
						'<a href="' . get_permalink( $post_id ) . '" class="button" title="' . esc_html__( 'View Page', 'aardvark' ) . ' "target="_blank">' . esc_html__( 'View Page', 'aardvark' ) . '</a>' .
						'<a href="' . get_admin_url( null, 'post.php?post=' . $post_id . '&action=edit' ) . '" class="button" title="' . esc_html__( 'Edit Page', 'aardvark' ) . '" target="_blank">' . esc_html__( 'Edit Page', 'aardvark' ) . '</a>' .
					'</div>';
				}
				
			} else {
			
				if ( isset( $data['import_page'] ) ) {
					$success_message .= '<p> ' . esc_html__( 'This demo has already been imported. Any pages that were deleted have been reimported.', 'aardvark' ) . '</p>';
				}
				
			}

			if ( ! $imported ) {
				$this->error .= '<p> ' . esc_html__( 'You have not selected any content to be imported.', 'aardvark' ) . '</p>';
			}
			
			if ( '' == $this->error ) {
				return  array(
					'message' => $success_message,
				);
			} else {
				return new WP_Error( '__k__', $this->error );
			}
			
		}

		/**
		* Show progress
		*
		*/
		public function set_progress( $post_id, $data ) {
			if ( $post_id ) {
				if ( ! isset( $data['progress'] ) ) {
					if ( 0 == $this->done_processes ) {
						$data['progress'] = 1;
					} else {
						$data['progress'] = floor( $this->done_processes / $this->processes * 100 );
					}
				}
				set_transient( 'ghostpool_import_' . floatval( $post_id ), $data, 60 * 60 );
			}
		}
	
		/**
		* Activate plugins
		*
		*/	
		public function activate_plugins( $set_data ) {
			if ( isset( $set_data['plugins'] ) && ! empty( $set_data['plugins'] ) ) {
				foreach( $set_data['plugins'] as $plugin ) {
					if ( ! class_exists( 'GhostPool_Addons_Manager' ) ) {
						require_once( get_theme_file_path( '/lib/framework/theme-setup/class-addons.php' ) );
					}
					$msg = '';
					$plugin_nice_name = ucfirst( str_replace( array( '_','-' ), ' ', $plugin ) );

					$this->set_progress( $this->progress_pid, array(
						'text' => sprintf( esc_html__( 'Installing Plugin: %s', 'aardvark' ), $plugin_nice_name ),
						'plugins' => $set_data['plugins'],
					) );

					if ( ! GhostPool_Addons_Manager()->is_plugin_installed( $plugin ) ) {
						$install = GhostPool_Addons_Manager()->do_plugin_install( $plugin, false );
						if ( isset( $install['error'] ) ) {
							$this->error .= '<p>' . $plugin_nice_name . ': ' . $install['error'] . '</p>';
						}
						$msg = sprintf( esc_html__( 'Installed Plugin %s', 'aardvark' ), $plugin_nice_name );
						$this->messages[] = $msg;
					}

					if ( ! GhostPool_Addons_Manager()->is_plugin_active( $plugin ) ) {
						$activate = GhostPool_Addons_Manager()->do_plugin_activate( $plugin, false );
						if ( isset( $activate['error'] ) ) {
							$this->error .= '<p>' . $plugin_nice_name . ': ' .  $activate['error'] . '</p>';
						}
						$msg = sprintf( esc_html__( 'Activated Plugin %s', 'aardvark' ), $plugin_nice_name );
						$this->messages[] = $msg;
					}
				
					if ( $msg ) {
						$this->set_progress( $this->progress_pid, array(
							'text' => $msg,
							'plugins' => $set_data['plugins'],
						) );
					}
				
					// Update permalink structure if set to plain
					if ( ! get_option( 'permalink_structure' ) ) {
						update_option( 'permalink_structure', '/%category%/%postname%/' );
					}
	
					$this->done_processes ++;
				
				}
			}
		}
	
		/**
		* Import content
		*
		*/
		public function import_content( $file = '', $force_attachments = false ) {
			$import = new WP_Import();
			$xml = get_theme_file_path( '/lib/framework/theme-setup/importer/demo-files/' . $file );
			if ( true == $force_attachments ) {
				$import->fetch_attachments = true;
			} else {
				$import->fetch_attachments = ( $_POST && array_key_exists( 'attachments', $_POST ) && $_POST['attachments'] ) ? true : false;
			}
			ob_start();
			$import->import( $xml );
			ob_end_clean();
		}

		/**
		* Import sidebars
		*
		*/	
		public function import_sidebars( $data ) {		
			if ( $data ) {
				$prev_value  = get_option( 'ghostpool_sidebars' );
				if ( ! empty( $prev_value ) ) {
					$data = array_merge( $data, $prev_value );
				}
				update_option( 'ghostpool_sidebars', $data );
			}	
		}

		/**
		* Import widgets
		*
		*/	
		public function import_widget_data( $json_data ) {

			$json_data    = json_decode( $json_data, true );
			$sidebar_data = $json_data[0];
			$widget_data  = $json_data[1];

			// Prepare widgets table
			$widgets = array();
			foreach( $widget_data as $k_w => $widget_type ) {
				if ( $k_w ) {
					$widgets[ $k_w ] = array();
					foreach( $widget_type as $k_wt => $widget ) {
						if ( is_int( $k_wt ) ) {
							$widgets[ $k_w ][ $k_wt ] = 1;
						}
					}
				}
			}

			// Sidebars
			foreach( $sidebar_data as $title => $sidebar ) {
				$count = count( $sidebar );
				for ( $i = 0; $i < $count; $i ++ ) {
					$widget               = array();
					$widget['type']       = trim( substr( $sidebar[ $i ], 0, strrpos( $sidebar[ $i ], '-' ) ) );
					$widget['type-index'] = trim( substr( $sidebar[ $i ], strrpos( $sidebar[ $i ], '-' ) + 1 ) );
					if ( ! isset( $widgets[ $widget['type'] ][ $widget['type-index'] ] ) ) {
						unset( $sidebar_data[ $title ][ $i ] );
					}
				}
				$sidebar_data[ $title ] = array_values( $sidebar_data[ $title ] );
			}

			// Widgets
			foreach( $widgets as $widget_title => $widget_value ) {
				foreach( $widget_value as $widget_key => $widget_value ) {
					$widgets[ $widget_title ][ $widget_key ] = $widget_data[ $widget_title ][ $widget_key ];
				}
			}

			$sidebar_data = array( array_filter( $sidebar_data ), $widgets );
			$this->parse_import_data( $sidebar_data );
		}

		/**
		* Import Widgets
		* @param $json_data
		* http://wordpress.org/plugins/widget-settings-importexport/
		*/
		public function parse_import_data( $import_array ) {
			$sidebars_data = $import_array[0];
			$widget_data   = $import_array[1];

			$current_sidebars = get_option( 'sidebars_widgets' );
			$new_widgets      = array();

			foreach( $sidebars_data as $import_sidebar => $import_widgets ) :

				$current_sidebars[ $import_sidebar ] = array();

				foreach( $import_widgets as $import_widget ) :
					//if the sidebar exists
					if ( isset( $current_sidebars[ $import_sidebar ] ) ) :

						$title               = trim( substr( $import_widget, 0, strrpos( $import_widget, '-' ) ) );
						$index               = trim( substr( $import_widget, strrpos( $import_widget, '-' ) + 1 ) );
						$current_widget_data = get_option( 'widget_' . $title );
						$new_widget_name     = self::get_new_widget_name( $title, $index );
						$new_index           = trim( substr( $new_widget_name, strrpos( $new_widget_name, '-' ) + 1 ) );
						if ( ! empty( $new_widgets[ $title ] ) && is_array( $new_widgets[ $title ] ) ) {
							while ( array_key_exists( $new_index, $new_widgets[ $title ] ) ) {
								$new_index ++;
							}
						}
						if (! $current_widget_data ) {
							$current_widget_data = array();
						}
						$current_sidebars[ $import_sidebar ][] = $title . '-' . $new_index;
						if ( array_key_exists( $title, $new_widgets ) ) {
							$new_widgets[ $title ][ $new_index ] = $widget_data[ $title ][ $index ];
							$multiwidget                         = $new_widgets[ $title ]['_multiwidget'];
							unset( $new_widgets[ $title ]['_multiwidget'] );
							$new_widgets[ $title ]['_multiwidget'] = $multiwidget;
						} else {
							$current_widget_data[ $new_index ] = $widget_data[ $title ][ $index ];
							$current_multiwidget               = isset( $current_widget_data['_multiwidget'] ) ? $current_widget_data['_multiwidget'] : '';
							$new_multiwidget                   = isset( $widget_data[ $title ]['_multiwidget'] ) ? $widget_data[ $title ]['_multiwidget'] : false;
							$multiwidget                       = ( $current_multiwidget != $new_multiwidget ) ? $current_multiwidget : 1;
							unset( $current_widget_data['_multiwidget'] );
							$current_widget_data['_multiwidget'] = $multiwidget;
							$new_widgets[ $title ]               = $current_widget_data;
						}
					endif;
				endforeach;
			endforeach;
			if ( isset( $new_widgets ) && isset( $current_sidebars ) ) {
				update_option( 'sidebars_widgets', $current_sidebars );
				foreach( $new_widgets as $title => $content ) {
					$content = apply_filters( 'widget_data_import', $content, $title );
					update_option( 'widget_' . $title, $content );
				}

				return true;
			}

			return false;
		}

		/**
		* Get new widget name
		* http://wordpress.org/plugins/widget-settings-importexport/
		*
		*/
		public function get_new_widget_name( $widget_name, $widget_index ) {
			$current_sidebars = get_option( 'sidebars_widgets' );
			$all_widget_array = array();
			foreach( $current_sidebars as $sidebar => $widgets ) {
				if ( ! empty( $widgets ) && is_array( $widgets ) && $sidebar != 'wp_inactive_widgets' ) {
					foreach( $widgets as $widget ) {
						$all_widget_array[] = $widget;
					}
				}
			}
			while ( in_array( $widget_name . '-' . $widget_index, $all_widget_array ) ) {
				$widget_index ++;
			}
			$new_widget_name = $widget_name . '-' . $widget_index;

			return $new_widget_name;
		}

		/**
		* Import theme settings
		*
		*/	
		public function import_options( $file = '' ) {
		
			if ( '' == $file ) {
				return;
			}

			$import_file = get_template_directory_uri() . '/lib/framework/theme-setup/importer/demo-files/' . $file . '.txt';
		
			if ( $import_file ) {
					
				$options = json_decode( ghostpool_fs_get_contents( $import_file ), true );

				if ( $file == 'landing-page/options' ) {
					$directory = '';
				} else {
					$directory = trailingslashit( substr( $file, 0, strpos( $file, '/' ) ) );
				}	
				$demo_template_directory = 'https://aardvark.ghostpool.com/' . $directory . 'wp-content/themes/aardvark/';
							
				foreach( $options as $key => $value ) {

					// Replace demo URLs with current site URLs
					if ( isset( $value['standard'] ) && strpos( $value['standard'], $demo_template_directory ) !== false ) {
						$value['standard'] = str_replace( $demo_template_directory, trailingslashit( get_template_directory_uri() ), $value['standard'] );
					}
					
					if ( isset( $value['retina'] ) && strpos( $value['retina'], $demo_template_directory ) !== false ) {
						$value['retina'] = str_replace( $demo_template_directory, trailingslashit( get_template_directory_uri() ), $value['retina'] );
					}
			
					$options[ $key ] = $value;
				
				}
				
				update_option( 'ghostpool_aardvark_settings', $options );

			}

		}

		/**
		* Import Ultimate Reviewer settings
		*
		*/	
		public function import_gpur_settings( $file = '' ) {
			if ( '' == $file ) {
				return;
			}
			$file_path 	= get_theme_file_path( '/lib/framework/theme-setup/importer/demo-files/' . $file . '.txt' );
			$file_data 	= ghostpool_fs_get_contents( $file_path );
			if ( $file_data ) {
				$data = json_decode( $file_data, true );
				update_option( 'gpur_settings', $data );
			}
		}
				
		/**
		* Import LayerSlider
		*
		*/
		public function import_layerslider( $file ) {
			if ( defined( 'LS_ROOT_PATH' ) ) {
				include( LS_ROOT_PATH . '/classes/class.ls.importutil.php' );
				$import = new LS_ImportUtil( $file );
			}
		}

		/**
		*  Import BuddyPress data
		*
		*/
		public function import_buddypress_data() {
		
			include_once( get_theme_file_path( '/lib/framework/theme-setup/importer/demo-files/buddypress-data/helpers.php' ) );
			
			include_once( get_theme_file_path( '/lib/framework/theme-setup/importer/demo-files/buddypress-data/process.php' ) );

			// Import users
			if ( ! bpdd_is_imported( 'users', 'users' ) ) {
				$users = bpdd_import_users();
				bpdd_update_import( 'users', 'users' );
			}
			if ( ! bpdd_is_imported( 'users', 'xprofile' ) ) {
				$profile = bpdd_import_users_profile();
				bpdd_update_import( 'users', 'xprofile' );
			}
			if ( ! bpdd_is_imported( 'users', 'friends' ) ) {
				$friends = bpdd_import_users_friends();
				bpdd_update_import( 'users', 'friends' );
			}
			if ( ! bpdd_is_imported( 'users', 'messages' ) ) {
				$messages = bpdd_import_users_messages();
				bpdd_update_import( 'users', 'messages' );
			}
			if ( ! bpdd_is_imported( 'users', 'activity' ) ) {
				$activity = bpdd_import_users_activity();
				bpdd_update_import( 'users', 'activity' );
			}

			// Import groups
			if ( ! bpdd_is_imported( 'groups', 'groups' ) ) {
				$groups = bpdd_import_groups();
				bpdd_update_import( 'groups', 'groups' );
			}
			if ( ! bpdd_is_imported( 'groups', 'members' ) ) {
				$g_members = bpdd_import_groups_members();
				bpdd_update_import( 'groups', 'members' );
			}
			if ( ! bpdd_is_imported( 'groups', 'activity' ) ) {
				$g_activity = bpdd_import_groups_activity();
				bpdd_update_import( 'groups', 'activity' );
			}
					
		}
	
		/**
		* Import BP profile search fields
		*
		*/		
		public function import_bp_fields( $bp_fields ) {
		
			if ( ! function_exists( 'bp_is_active' ) || ! bp_is_active( 'xprofile' ) ) {
				return;
			}

			$imported_ids = array();
			$imported_labels = array();
			$imported_descs = array();
			$imported_modes = array();
			$i = 0;
			
			foreach( $bp_fields as $field ) {

				$i++;
				if ( ! xprofile_get_field_id_from_name( $field['name'] ) ) {
					$id = xprofile_insert_field(
						array(
							'field_group_id' => 1,
							'name'           => $field['name'],
							'can_delete'     => $field['can_delete'],
							'field_order'    => $i,
							'is_required'    => $field['is_required'],
							'type'           => $field['type'],
						)
					);
					$imported_ids[] = $id;
					$imported_labels[] = $field['name'];
					$imported_descs[] = $field['desc'];
					$imported_modes[] = $field['mode'];

					if ( $id && isset( $field['options'] ) && ! empty( $field['options'] ) ) {
						$j = 0;
						foreach( $field['options'] as $option ) {
							$j++;
							xprofile_insert_field( array(
								'field_group_id' => 1,
								'parent_id'      => $id,
								'type'           => $field['type'],
								'name'           => $option,
								'option_order'   => $j,
							) );
						}
					}
				}
			}

			$this->import_bps_form( $imported_ids, $imported_labels, $imported_descs, $imported_modes );

		}

		/**
		* Import BP profile search form with new field IDs
		*
		*/		
		public function import_bps_form( $imported_ids, $imported_labels, $imported_descs, $imported_modes ) {
		
			if ( ! empty( $imported_ids ) ) {

				$field_code = array();
				foreach( $imported_ids as $imported_id ) {
					$field_code[] = 'field_' . $imported_id;
				}
			
				$bp_pages = get_option( 'bp-pages' );
				$member_page_id = (int) $bp_pages['members'];
				
				if ( ! post_type_exists( 'bps_form' ) ) {
					return;
				}
	
				$bps_form_args = array(
					'post_status' => 'draft',
					'post_type'   => 'bps_form',
					'post_title'  => 'Member Search',
				);
				$post_id = wp_insert_post( $bps_form_args );
											
				$new_value = array(
					'field_code' => $field_code,
					'field_label' => $imported_labels,
					'field_desc' => $imported_descs,
					'field_mode' => $imported_modes,
					'method' => 'POST',
					'action' => $member_page_id,
					'directory' => 'Yes',
					'template' => 'members/bps-form-default',
					'template_options' => array(
						'members/bps-form-default' => array(
							'header' => esc_html__( 'Member Search', 'aardvark' ),
							'toggle' => 'Enabled',
							'theme' => 'base',
							'collapsible' => 'Yes',
							'button' => esc_html__( 'Hide/Show Form', 'aardvark' ),
						),	
					),
				);
			
				update_post_meta( $post_id, 'bps_options', $new_value );
				update_post_meta( $post_id, 'ghostpool_import', '1' );

			}
		}

		/**
		* Import PMPro membership levels
		*
		*/		
		public function import_pmpro_levels() {
	
			global $wpdb;
	
			$current_levels = pmpro_getAllLevels( false, true );
			
			if ( ! $current_levels ) {		
		
				$levels = array( 
					array( 
						'id' => '1',
						'name' => 'Basic',
						'description' => 'Basic level access.',
						'confirmation' => '',
						'initial_payment' => '0.00',
						'billing_amount' => '10.00',
						'cycle_number' => '1',
						'cycle_period' => 'Month',
						'billing_limit' => '0',
						'trial_amount' => '0.00',
						'trial_limit' => '1',
						'expiration_number' => '0',
						'expiration_period' => '',
						'allow_signups' => 1,
					),			
					array( 
						'id' => '2',
						'name' => 'Standard',
						'description' => 'Standard level access.',
						'confirmation' => '',
						'initial_payment' => '0.00',
						'billing_amount' => '35.00',
						'cycle_number' => '1',
						'cycle_period' => 'Month',
						'billing_limit' => '0',
						'trial_amount' => '0.00',
						'trial_limit' => '0',
						'expiration_number' => '0',
						'expiration_period' => '',
						'allow_signups' => 1,
					),			
					array( 
						'id' => '3',
						'name' => 'Pro',
						'description' => 'Pro level access.',
						'confirmation' => '',
						'initial_payment' => '0.00',
						'billing_amount' => '100.00',
						'cycle_number' => '1',
						'cycle_period' => 'Month',
						'billing_limit' => '0',
						'trial_amount' => '0.00',
						'trial_limit' => '0',
						'expiration_number' => '0',
						'expiration_period' => '',
						'allow_signups' => 1,
					),
				);
					
				if ( ! empty( $levels ) ) {

					foreach( $levels as $level ) {

						$wpdb->insert( $wpdb->pmpro_membership_levels, 
							array( 
								'id' => $level['id'],	
								'name' => $level['name'],
								'description' => $level['description'],
								'confirmation' => $level['confirmation'],
								'initial_payment' => $level['initial_payment'],
								'billing_amount' => $level['billing_amount'],
								'cycle_number' => $level['cycle_number'],
								'cycle_period' => $level['cycle_period'],
								'billing_limit' => $level['billing_limit'],
								'trial_amount' => $level['trial_amount'],
								'trial_limit' => $level['trial_limit'],
								'expiration_number' => $level['expiration_number'],
								'expiration_period' => $level['expiration_period'],
								'allow_signups' => $level['allow_signups'],
							),
							array(
								'%d',		//id
								'%s',		//name
								'%s',		//description
								'%s',		//confirmation
								'%f',		//initial_payment
								'%f',		//billing_amount
								'%d',		//cycle_number
								'%s',		//cycle_period
								'%d',		//billing_limit
								'%f',		//trial_amount
								'%d',		//trial_limit
								'%d',		//expiration_number
								'%s',		//expiration_period
								'%d',		//allow_signups
							)
						);
			
					}		
				}	
			
			}
		}
		
		/**
		* Import Youzer options
		*
		*/		
		public function import_youzer_options() {
			
			update_option( 'yz_group_header_layout', 'hdr-v2' );
			update_option( 'logy_login_form_enable_header', 'off' );
			update_option( 'logy_login_form_title', 'Signup to the Community' );
			
			update_option( 'logy_login_form_layout', 'logy-field-v12' );
			update_option( 'logy_login_fields_format', 'logy-border-rounded' );
				
			update_option( 'logy_login_actions_layout', 'logy-actions-v4' );
			update_option( 'logy_login_btn_format', 'logy-border-rounded' );

			$logy_login_title_color = get_option( 'logy_login_title_color' );
			$logy_login_title_color['color'] = '#333333';
			update_option( 'logy_login_title_color', $logy_login_title_color );
			
			$logy_login_cover_title_bg_color = get_option( 'logy_login_cover_title_bg_color' );
			$logy_login_cover_title_bg_color['color'] = '#eeeeee';
			update_option( 'logy_login_cover_title_bg_color', $logy_login_cover_title_bg_color );
			
			$logy_login_submit_bg_color = get_option( 'logy_login_submit_bg_color' );
			$logy_login_submit_bg_color['color'] = '#fb386f';
			update_option( 'logy_login_submit_bg_color', $logy_login_submit_bg_color );
			
			$logy_login_regbutton_bg_color = get_option( 'logy_login_regbutton_bg_color' );
			$logy_login_regbutton_bg_color['color'] = '#333333';
			update_option( 'logy_login_regbutton_bg_color', $logy_login_regbutton_bg_color );

		}

		/**
		* Get content with "ghostpool_import" meta key
		*
		*/					
		private function get_imported_posts() {
			$args  = array(
				'post_type'  => array( 'post', 'page' ),
				'posts_per_page' => -1,
				'meta_query' => array(
					array(
						'key'   => 'ghostpool_import',
						'value' => '1',
					),
					array(
						'key'     => '_ghostpool_imported',
						'compare' => 'NOT EXISTS',
					),
				),
			);
			$query = new WP_Query( $args );
			return $query->posts;
		}
		
		/**
		* Get homepage from current demo
		*
		*/		
		private function get_demo_homepage() {
			$posts = $this->get_imported_posts();
			foreach( $posts as $post ) {
				if ( 'page' === get_post_type( $post->ID ) ) {
					$this->posts_imported[ $post->ID ] = $post;
				}
			}
		}

		/**
		* Update WPBakery Page Builder row background urls
		*
		*/
		public function update_row_bgs( $post ) {
			global $wpdb;
		
			// Get old URLs from post meta
			if ( preg_match_all( '/background-image.*?url\((.*?)\)/mi', get_post_meta( $post->ID, '_wpb_shortcodes_custom_css', true ), $matches ) ) {			
				$old_urls = array();
				foreach( $matches[0] as $match ) {
					$old_urls[] = $match;
				}
			}	
		
			// Get new background URLs from post content
			if ( preg_match_all( '/background-image.*?url\((.*?)\)/mi', $post->post_content, $matches ) ) {
				$new_urls = array();
				foreach( $matches[0] as $match ) {
					$new_urls[] = $match;
				}
			}	
		
			// Combine old and new URLs into a single array
			if ( isset( $old_urls ) && is_array( $old_urls ) && isset( $new_urls ) && is_array( $new_urls ) ) {
				$urls = array_combine( $old_urls, $new_urls ); 
			}
					
			// Replace old URLs with new URLs
			if ( isset( $urls ) && is_array( $urls ) ) {
				foreach( $urls as $old_url => $new_url ) {
					$wpdb->query( $wpdb->prepare( "UPDATE $wpdb->postmeta SET meta_value = REPLACE( meta_value, %s, %s )", $old_url, $new_url ) );
				}
			}
						
		}

		/**
		* Update WPBakery Page Builder RTL rows
		*
		*/
		public function update_rtl_rows( $post ) {

			$search = array( 'background_position="right center"', 'text_align:left', 'align="left"' );
			$replace = array( 'background_position="left center"', 'text_align:right', 'align="right"' );

			$post_content = str_replace( $search, $replace, $post->post_content );
			
			$update_post = array(
				'ID' => $post->ID,
				'post_content' => $post_content,
			);

			wp_update_post( $update_post );
	
		}
			
		/**
		* Set default plugin pages
		*
		*/
		public function set_default_plugin_pages() {
		
			// Set WooCommerce pages
			$wc_shop_page = get_page_by_title( 'Shop' );
			if ( ! empty( $wc_shop_page ) && get_option( 'woocommerce_shop_page_id' ) == '' ) {
				update_option( 'woocommerce_shop_page_id', $wc_shop_page->ID );
			}	
			$wc_cart_page = get_page_by_title( 'Cart' );
			if ( ! empty( $wc_cart_page ) && get_option( 'woocommerce_cart_page_id' ) == '' ) {
				update_option( 'woocommerce_cart_page_id', $wc_cart_page->ID );
			}
			$wc_checkout_page = get_page_by_title( 'Checkout' );
			if ( ! empty( $wc_checkout_page ) && get_option( 'woocommerce_checkout_page_id' ) == '' ) {
				update_option( 'woocommerce_checkout_page_id', $wc_checkout_page->ID );
			}
			$wc_account_page = get_page_by_title( 'My account' );
			if ( ! empty( $wc_account_page ) && get_option( 'woocommerce_myaccount_page_id' ) == '' ) {
				update_option( 'woocommerce_myaccount_page_id', $wc_account_page->ID );
			}
			
			// Set Sensei pages		
			$sensei_courses_page = get_page_by_title( 'Courses' );
			if ( ! empty( $sensei_courses_page ) && get_option( 'woothemes-sensei_courses_page_id' ) == 0 ) {
				update_option( 'woothemes-sensei_courses_page_id', $sensei_courses_page->ID );
			}	
			$sensei_my_courses_page = get_page_by_title( 'My Courses' );
			if ( ! empty( $sensei_my_courses_page ) && get_option( 'woothemes-sensei_user_dashboard_page_id' ) == 0 ) {	
				update_option( 'woothemes-sensei_user_dashboard_page_id', $sensei_my_courses_page->ID );
			}
		
			// Set Paid Membership Pro pages
			$pmp_account_page = get_page_by_title( 'Membership Account' );
			if ( ! empty( $pmp_account_page ) && get_option( 'pmpro_account_page_id' ) == '' ) {	
				update_option( 'pmpro_account_page_id', $pmp_account_page->ID );
			
			}
			$pmp_billing_page = get_page_by_title( 'Membership Billing' );
			if ( ! empty( $pmp_billing_page ) && get_option( 'pmpro_billing_page_id' ) == '' ) {	
				update_option( 'pmpro_billing_page_id', $pmp_billing_page->ID );
			
			}
			$pmp_cancel_page = get_page_by_title( 'Membership Cancel' );
			if ( ! empty( $pmp_cancel_page ) && get_option( 'pmpro_cancel_page_id' ) == '' ) {	
				update_option( 'pmpro_cancel_page_id', $pmp_cancel_page->ID );
			
			}		
			$pmp_checkout_page = get_page_by_title( 'Membership Checkout' );
			if ( ! empty( $pmp_checkout_page ) && get_option( 'pmpro_checkout_page_id' ) == '' ) {	
				update_option( 'pmpro_checkout_page_id', $pmp_checkout_page->ID );
			
			}	
			$pmp_confirmation_page = get_page_by_title( 'Membership Confirmation' );
			if ( ! empty( $pmp_confirmation_page ) && get_option( 'pmpro_confirmation_page_id' ) == '' ) {	
				update_option( 'pmpro_confirmation_page_id', $pmp_confirmation_page->ID );
			
			}	
			$pmp_invoice_page = get_page_by_title( 'Membership Invoice' );
			if ( ! empty( $pmp_invoice_page ) && get_option( 'pmpro_invoice_page_id' ) == '' ) {	
				update_option( 'pmpro_invoice_page_id', $pmp_invoice_page->ID );
			
			}	
			$pmp_levels_page = get_page_by_title( 'Membership Levels' );
			if ( ! empty( $pmp_levels_page ) && get_option( 'pmpro_levels_page_id' ) == '' ) {	
				update_option( 'pmpro_levels_page_id', $pmp_levels_page->ID );
			
			}		
				
		}

		/**
		* Event manager fixes
		*
		*/		
		public function event_manager_fixes() {
		
			if ( ! function_exists( 'em_init' ) ) {
				return;
			}	

			// Fix for capabilities being cleared
			if ( ! user_can( 'administrator', 'publish_events' ) ) {
	
				global $wp_roles;

				$caps = array( 'publish_events', 'delete_others_events', 'edit_others_events', 'manage_others_bookings', 'publish_recurring_events', 'delete_others_recurring_events', 'edit_others_recurring_events', 'publish_locations', 'delete_others_locations',	'delete_locations', 'edit_others_locations', 'delete_event_categories', 'edit_event_categories' );

				$roles = array( 'administrator', 'editor' );

				foreach( $roles as $user_role ) {
					foreach( $caps as $cap ) {
						$wp_roles->add_cap( $user_role, $cap );
					}
				}
	
				$loose_caps = array( 'manage_bookings', 'upload_event_images', 'delete_events', 'edit_events', 'read_private_events', 'delete_recurring_events', 'edit_recurring_events', 'edit_locations', 'read_private_locations', 'read_others_locations' );

				$roles = array( 'administrator', 'editor', 'contributor', 'author', 'subscriber' );

				foreach( $roles as $user_role ) {
					foreach( $loose_caps as $cap ) {
						$wp_roles->add_cap( $user_role, $cap );
					}
				}
		
			}		
	
		}
								
		private function post_process_posts( $set_data ) {

			// Set default plugin pages
			$this->set_default_plugin_pages();
			
			// Event manager fixes
			$this->event_manager_fixes();
					
			$posts = $this->get_imported_posts();

			foreach( $posts as $post ) {

				do_action( 'ghostpool_import_before_process', $post, $this );
			
				$this->update_row_bgs( $post );
				
				if ( is_rtl() ) {
					$this->update_rtl_rows( $post );
				}
				
				do_action( 'ghostpool_import_after_process', $post, $this );

				add_post_meta( $post->ID, '_ghostpool_imported', '1' );

			}
			wp_reset_postdata();

			delete_post_meta_by_key( 'ghostpool_import' );
				
		}
				
		public function get_post_by_slug( $slug ) {
			global $wpdb;
			return $wpdb->get_var( $wpdb->prepare( "SELECT ID FROM $wpdb->posts WHERE post_name = %s", $slug ) );
		}
		
		private function set_success_message( $msg ) {
			return $msg;
		}
		
		public function do_import() {
			if ( array_key_exists( 'ghostpool_import_nonce', $_POST ) ) {

				if ( wp_verify_nonce( $_POST['ghostpool_import_nonce'], 'import_nonce' ) ) {

					if ( isset( $_POST['import_demo'] ) ) {

						$demos   = self::get_demo_sets();
						$current_set = $_POST['import_demo'];
						$set_data = $demos[ $current_set ];

						if ( ! array_key_exists( $current_set, $demos ) ) {
							$this->error .= esc_html__( 'Something went wrong with the data sent. Please try again.', 'aardvark' );
						}

						$data = array();
						foreach( $_POST as $key => $value ) {
							if ( is_array( $value ) && in_array( $current_set, $value ) ) {
								$data[ $key ][] = $current_set;
							}
						}

						$response = $this->process_import( array(
							'set_data' => $set_data,
							'pid' => false,
							'data' => $data,
						) );

						if ( is_wp_error( $response ) ) {
							$this->error .= $response->get_error_message();
						} else {
							$this->data_imported = true;
						}
					
					} else {

						// Importer classes
						if ( ! defined( 'WP_LOAD_IMPORTERS' ) ) {
							define( 'WP_LOAD_IMPORTERS', true );
						}

						if ( ! class_exists( 'WP_Importer' ) ) {
							require_once ABSPATH . 'wp-admin/includes/class-wp-importer.php';
						}

						if ( ! class_exists( 'WP_Import' ) ) {
							require_once( get_theme_file_path( '/lib/framework/theme-setup/importer/wordpress-importer.php' ) );
						}

						if ( class_exists( 'WP_Importer' ) && class_exists( 'WP_Import' ) ) {
							switch ( $_POST['import'] ) {
							}
						}
						$this->data_imported = true;
						
					}
				}
			}
		}

		/**
		* Import menu locations
		*
		* @param array $locations Menu locations and names
		*/
		function import_menu_location( $data = array() ) {

			$menus = wp_get_nav_menus();

			foreach( $data as $key => $value ) {
				foreach( $menus as $menu ) {
					if ( $menu->slug == $value ) {
						$data[ $key ] = absint( $menu->term_id );
					}
				}
			}
			set_theme_mod( 'nav_menu_locations', $data );
		}
		
		/**
		* Generate demo importer page
		*
		*/
		public function generate_boxes_html() { ?>
		
			<div class="theme-browser">
		
				<div class="themes">

					<?php
			
					$demos = self::get_demo_sets();

					foreach( $demos as $key => $demo ) : ?>

						<div class="theme" <?php if ( isset( $demo['slug'] ) ) { echo 'data-slug="' . $demo['slug'] . '"'; } ?>>
	
							<?php if ( isset( $demo['slug'] ) && $this->get_post_by_slug( $demo['slug'] ) ) : ?>
								<div class="gp-imported-label"></div>
							<?php endif; ?>
						
							<div class="theme-screenshot">
								<img src="<?php echo esc_attr( $demo['img'] ); ?>">
							</div>
						
							<div class="theme-id-container">
							
								<div class="gp-demo-left">	
							
									<h2 class="theme-name"><?php echo esc_attr( $demo['name'] ); ?></h2> 
								
									<div class="gp-demo-checkboxes">
									
										<?php if ( isset( $demo['page'] ) ) : ?>
											<label><input type="checkbox" name="import_page[]" checked value="<?php echo esc_attr( $key ); ?>" class="check-page"> <?php esc_html_e( 'Import Pages' , 'aardvark' ); ?></label>
										<?php endif; ?>

										<?php if ( isset( $demo['extra'] ) && ! empty( $demo['extra'] ) ) : ?>
											<?php foreach( $demo['extra'] as $extra ) : ?>
												<?php
												if ( $extra['id'] == 'sensei' && ! function_exists( 'is_sensei' ) ) {
													$checked = ' disabled';
													$strike_start = '<s>';
													$strike_end = '</s>';
												} elseif ( $extra['id'] == 'products' && ! function_exists( 'is_woocommerce' ) ) {
													$checked = ' disabled';
													$strike_start = '<s>';
													$strike_end = '</s>';
												} elseif ( isset( $extra['checked'] ) && $extra['checked'] ) {
													$checked = ' checked="checked"';
													$strike_start = '';
													$strike_end = '';
												} else {
													$checked = '';
													$strike_start = '';
													$strike_end = '';
												}
												?>
												<label>
													<input type="checkbox" name="import_<?php echo esc_attr( $extra['id'] );?>[]" value="<?php echo esc_attr( $key ); ?>" class="check-page"<?php echo esc_attr( $checked ); ?>><?php echo wp_kses_post( $strike_start ); ?> <?php echo esc_attr( $extra['name'] ); ?><?php echo wp_kses_post( $strike_end ); ?>
												</label>
											<?php endforeach; ?>
										<?php endif; ?>

										<?php if ( isset( $demo['widgets'] ) ) : ?>
											<label><input type="checkbox" name="import_widgets[]" checked value="<?php echo esc_attr( $key ); ?>"> <?php esc_html_e( 'Import Widgets' , 'aardvark' ); ?></label>
										<?php endif; ?>

										<?php if ( isset( $demo['bp_fields'] ) ) : ?>
											<label><input type="checkbox" name="import_bp_fields[]" value="<?php echo esc_attr( $key ); ?>"<?php if ( ! function_exists( 'bp_is_active' ) OR ! bp_is_active( 'xprofile' ) ) { ?> disabled<?php } else { ?> checked<?php } ?>> <?php if ( ! function_exists( 'bp_is_active' ) OR ! bp_is_active( 'xprofile' ) ) { ?><s><?php } ?><?php esc_html_e( 'Import BuddyPress Profile Fields' , 'aardvark' ); ?><?php if ( ! function_exists( 'bp_is_active' ) OR ! bp_is_active( 'xprofile' ) ) { ?></s><?php } ?></label>
										<?php endif; ?>

										<?php if ( isset( $demo['layerslider'] ) ) : ?>
											<label><input type="checkbox" name="import_layerslider[]" value="<?php echo esc_attr( $key ); ?>"<?php if ( ! class_exists( 'ZipArchive' ) ) { ?> disabled<?php } else { ?> checked<?php } ?>> <?php if ( ! class_exists( 'ZipArchive' ) ) { ?><s><?php } ?><?php esc_html_e( 'Import LayerSlider' , 'aardvark' ); ?><?php if ( ! class_exists( 'ZipArchive' ) ) { ?></s><?php } ?></label>
										<?php endif; ?>

										<?php if ( isset( $demo['levels'] ) ) : ?>
											<label><input type="checkbox" name="import_levels[]" value="<?php echo esc_attr( $key ); ?>"><?php esc_html_e( 'Import Membership Levels' , 'aardvark' ); ?></label>
										<?php endif; ?>
									
										<?php if ( isset( $demo['buddypress'] ) ) : ?>
											<label><input type="checkbox" name="import_buddypress[]" value="<?php echo esc_attr( $key ); ?>"<?php if ( ! function_exists( 'bp_is_active' ) ) { ?> disabled<?php } else { ?> checked<?php } ?>> <?php if ( ! function_exists( 'bp_is_active' ) ) { ?><s><?php } ?><?php esc_html_e( 'Import BuddyPress Data' , 'aardvark' ); ?><?php if ( ! function_exists( 'bp_is_active' ) ) { ?></s><?php } ?></label>
										<?php endif; ?>
									
										<?php if ( isset( $demo['options'] ) ) : ?>
											<label>
												<input type="checkbox" name="import_options[]" checked value="<?php echo esc_attr( $demo['options'] ); ?>"> <?php esc_html_e( 'Import Theme Options' , 'aardvark' ); ?>
												<?php 
												$extra_options_data = esc_html__( 'This will change some of your theme options. Make sure to export them first.', 'aardvark' );
												echo ' <span class="dashicons dashicons-editor-help tooltip-me" title="' . $extra_options_data . '"></span>';
												?>
											</label>	
										<?php endif; ?>

									
										<?php if ( isset( $demo['gpur_settings'] ) ) : ?>
											<label>
												<input type="checkbox" name="import_gpur_settings[]" checked value="<?php echo esc_attr( $demo['gpur_settings'] ); ?>"> <?php esc_html_e( 'Import Ultimate Reviewer Settings' , 'aardvark' ); ?>
												<?php 
												$extra_options_data = esc_html__( 'This will change your existing Ultimate Reviewer plugin settings.', 'aardvark' );
												echo ' <span class="dashicons dashicons-editor-help tooltip-me" title="' . $extra_options_data . '"></span>';
												?>
											</label>	
										<?php endif; ?>
										
										<?php if ( isset( $demo['plugins'] ) && ! empty( $demo['plugins'] ) ) : ?>
											<label>
												<input type="checkbox" name="activate_plugins[]" checked value="<?php echo esc_attr( $key ); ?>">
												<?php
												echo esc_html__( 'Activate Required Plugins', 'aardvark' );
												$extra_plugin_data = str_replace( '-', ' ', implode( ', ', $demo['plugins'] ) );
												$extra_plugin_data = ucwords( str_replace( '_', ' ', $extra_plugin_data ) );
												echo ' <span class="dashicons dashicons-editor-help tooltip-me" title="' . $extra_plugin_data . '"></span>';
												?>
											</label>
										<?php endif; ?>
									
										<?php if ( isset( $demo['notices'] ) ) {
									
											$has_notice = false;
											
											$notices = $demo['notices'];
								
											$output = '<strong>' . esc_html__( 'Activate the following plugins BEFORE importing the demo if required:', 'aardvark' ) . '</strong>';
										
											$output .=  '<ul>';
								
												if ( in_array( 'buddypress', $notices ) && ! function_exists( 'bp_is_active' ) ) {
													$output .= '<li>' . esc_html__( 'BuddyPress - to import BuddyPress demo data', 'aardvark' ) . '</li>';
													$has_notice = true;
												}
								
												if ( in_array( 'bp-profile-search', $notices ) && ! defined( 'BPS_FORM' ) ) {
													$output .= '<li>' . esc_html__( 'BP Profile Search  - to import BP search forms', 'aardvark' ) . '</li>';
													$has_notice = true;
												}
									
												if ( in_array( 'events-manager', $notices ) && ! function_exists( 'em_init' ) ) {
													$output .= '<li>' . esc_html__( 'Events Manager - to import events data', 'aardvark' ) . '</li>';
													$has_notice = true;
												}
								
												if ( in_array( 'woocommerce', $notices ) && ! function_exists( 'is_woocommerce' ) ) {
													$output .= '<li>' . esc_html__( 'WooCommerce - to import product demo data', 'aardvark' ) . '</li>';
													$has_notice = true;
												}	
								
												if ( in_array( 'sensei', $notices ) && ! function_exists( 'is_sensei' ) ) {
													$output .= '<li>' . esc_html__( 'Sensei - to import courses demo data', 'aardvark' ) . '</li>';
													$has_notice = true;
												} 
												
												if ( ! class_exists( 'ZipArchive' ) ) {
													$output .= '<li>' . esc_html__( 'ZipArchive is required for importing LayerSlider. Please contact your web host to enable it.', 'aardvark' ) . '</li>';
													$has_notice = true;
												}
								
											$output .= '</ul>';
										
											if ( true === $has_notice ) {
										
												echo '<span class="gp-demo-notices">' . $output . '</span>';
											
											}	
											
										} ?>		
									
										<small><?php esc_html_e( 'It is recommended to leave all default options checked to reproduce this demo.', 'aardvark' ); ?></small>
								
									</div>
							
								</div>

								<button type="submit" name="import_demo" value="<?php echo esc_attr( $key ); ?>" class="button button-primary gp-import-button"><?php esc_html_e( 'Import', 'aardvark' ); ?></button>
												
								<a href="<?php echo esc_attr( $demo['link'] ); ?>" class="button gp-preview-button" target="_blank"><?php esc_html_e( 'Preview' , 'aardvark' ); ?></a>
							
							</div>
						
						</div>

					<?php endforeach; ?>
			
				</div>

			</div>
		
			<?php
		}
				
	}
}
GhostPool_Importer::getInstance();